/*********************************************/
/* Example Program for ET-ADuC847 STAMP V1.0 */
/* MCU      : ADuC847(XTAL = 32.768 KHz)     */
/*          : Frequency Bus = 12.582912 MHz  */
/* Compiler : Keil C51 (V7.50)               */
/* Function : QC Test Hardware System Demo   */
/* Write By : ETT CO.,LTD.(May/2011)         */
/*********************************************/

/* include  section */
#include <aduc847.h>								// ADUC847 SFR : File
#include <stdio.h>                					// Prototype declarations for I/O functions

static bit Trig_Ready;								// 1/64mS Overflow Status

/* pototype  section */
void test_adc(void);
void test_dac(void);
void test_i2c(void);
void test_gpio_manual(void);
void test_gpio_auto(void);

void write_PCF8574(unsigned char ,unsigned char);	// Write Output PCF8574A
void  I2C_Delay(void);				   				// Delay Clock
void delay(unsigned long);							// Delay Time Function(1..4294967295)

/*--------------------------------------------
The main C function.  Program execution Here 
---------------------------------------------*/
void main (void) 
{  
  unsigned char ch;									// RS232 Get Byte
   
  PLLCON = 0x00;									// Initial Clock = 12.582912MHz

  /* Initial MCS51 Serial Port */
  /* Used Timer3 Gen. Baudrate */
  // Baud   = T3CON,T3FD
  // 9600   = 0x86,0x12
  // 19200  = 0x85,0x12
  // 38400  = 0x84,0x12
  // 57600  = 0x83,0x2D
  // 115200 = 0x82,0x2D
  // 230400 = 0x81,0x2D
  T3CON = 0x86;										// 9600BPS/12.58MHz(32.768KHz+PLL)
  T3FD  = 0x12;        
  SCON  = 0x50;  									// Serial Port Mode 1 (N,8,1)
  ES    = 0;     									// Disable Serial Interupt            
  TI    = 1;     									// Set TI to send First char of Keil-UART Function

  /* Print Message to RS232 */
  printf("\n\n\n");
  printf("*****************************\n");
  printf("*** ET-ADuC847 STAMP V1.0 ***\n"); 
  printf("*****************************\n");
  printf(">>>> QC SYSTEM SELF TEST <<<<\n");
  printf("1. Test ADC 24 Bit/8Ch\n");  
  printf("2. Test DAC 12 Bit\n");  
  printf("3. Test I2C PCF8574A(Output)\n");
  printf("4. Test GPIO(Output) Manual\n");
  printf("5. Test GPIO(Output) Auto\n");
  printf("*****************************\n");
  printf("\n");
  printf("Select Choice(1...5)...");

  while(1)											// Loop Continue
  {
    ch = getchar();  								// Get Choice
	switch (ch)
	{
	  case '1' : printf("\n\n");
	             printf("Test ADC 24 Bit/8Ch\n");
				 test_adc();
	             break;

	  case '2' : printf("\n\n");
	             printf("Test DAC 12 Bit\n");
				 test_dac();
	             break;

	  case '3' : printf("\n\n");
	             printf("Test I2C PCF8574A(Output)\n");
				 test_i2c();
	             break;	  

	  case '4' : printf("\n\n");
	             printf("Test GPIO(Output) Manual\n");	 
				 test_gpio_manual();
				 break;
	  case '5' : printf("\n\n");
	             printf("Test GPIO(Output) Auto\n");	 
				 test_gpio_auto();
				 break;
	}
  }  
}

// Demo Test ADC Unipolar Mode / 8 Channel
void test_adc(void)
{
  unsigned int val;		   							// ADC Result (HEX)
  float volt;										// ADC Result Volt
  
  //Configure ADC and Start Converting...  
  /* Start of Initial ADC */
  ADC0CON1  =  0x00;								// Reset All Config Bit
  ADC0CON1 &= ~0xC0;								// Full Buffer
  ADC0CON1 |=  0x20;								// Unipolar Mode
  ADC0CON1 &= ~0x07;								// Reset ADC Input Range
  ADC0CON1 |=  0x07;								// Select ADC Input Range = 0V-2.56V(Unipolar) 

  ADC0CON2  =   0x00;								// Reset All Bit
  ADC0CON2 &=  ~0xC0;          						// Select Vref = Internal 1.25V
  ADC0CON2 &=  ~0x0F;	            				// ADC Input = AIN1,AINCOM
  ADCMODE   =   0x00;								// Reset All Bit
  ADCMODE  |=   0x40;								// Enable Automatic Reject 60Hz Noise	
  ADCMODE  |=   0x20;								// Enable ADC0  
  ADCMODE  |=   0x03;								// Continue Conversion

  //Chop Enable Filter
  ADCMODE  &=  ~0x08;								// Enable CHOP Mode  
  SF        =   0x0D;								// Fadc=105.3Hz,Tadc=9.52mS,Tsettle=19.04mS 

  //Chop Disable Filter
  //ADCMODE  |=   0x08;								// Disable CHOP Mode  
  //SF        =   0x52;								// Fadc=49.85Hz,Tadc=20.2mS,Tsettle=60.06mS  
  /* End of Initial ADC */

  /* Print Message to RS232 */
  printf("\n\n\n\n\r");
  printf("\n=======================================");
  printf("\nET-STAMP ADuC847.....From ETT CO.,LTD");  
  printf("\nDemo ADC Unipolar AN?,COM = 0..1.250V");    
  printf("\nDevice Offset = %bX%bX%bX",OF0H,OF0M,OF0L);   
  printf("\nDevice Gain   = %bX%bX%bX",GN0H,GN0M,GN0L);   
  printf("\n=======================================");
  
  //Zero Scale Calibrate  
  printf("\n\n\r");
  printf("\nDevice Calibrate Zero Scale Value");    
  printf("\nInput Zero Scale VDC to AN1 & COM");  
  printf("\nPress Any Key to Continue.......");    
  RI = 0;  
  while (!RI);										// Wait Any Key
  
  ADCMODE &=  ~0x07;	            				// Reset Mode ADC Input
  ADCMODE |=   0x04;	            				// ADC Input = Internal Zero Scale
  RDY0 =  0; 
  while(!RDY0);

  ADCMODE &=  ~0x07;	            				// Reset Mode ADC Input
  ADCMODE |=   0x06;	            				// ADC Input = System Zero Scale
  RDY0 =  0; 
  while(!RDY0);

  //Full Scale Calibrate
  printf("\n\n\r");
  printf("\nDevice Calibrate Full Scale Value");    
  printf("\nInput Full Scale VDC to AN1 & COM");
  printf("\nPress Any Key to Continue.......");  
  RI = 0;
  while (!RI);										// Wait Any Key
  
  ADCMODE &=  ~0x07;	            				// Reset Mode ADC Input
  ADCMODE |=   0x05;	            				// ADC Input = Internal Full Scale
  RDY0 =  0; 
  while(!RDY0);

  ADCMODE &=  ~0x07;	            				// Reset Mode ADC Input
  ADCMODE |=   0x07;	            				// ADC Input = System Full Scale
  RDY0 =  0; 
  while(!RDY0);
  
  printf("\n\r");
  printf("\n=======================================");
  printf("\nDevice Offset = %bX%bX%bX",OF0H,OF0M,OF0L);   
  printf("\nDevice Gain   = %bX%bX%bX",GN0H,GN0M,GN0L);   
  printf("\n=======================================");
  printf("\n\n\r");

  //Normal Operation
  ADCMODE  &=  ~0x07;	            				// Reset Mode ADC Input
  ADCMODE  |=   0x03;								// Continue Conversion  
  while (1) 
  {
    //Get Result in 16Bit ADC0H + ADC0M ,Ignore ADC0L
    //AN1
	ADC0CON2 &=  ~0x0F;	            				// ADC Input = AIN1,AINCOM
	RDY0 =  0; 
  	while(!RDY0);	     							// Wait ADC Complete		
	val  = ((ADC0H<<8)&0xFF00);          			// Read MSB 8-Bits 
    val |= ((ADC0M<<0)&0x00FF);   					// Read LSB 8-Bits 
	volt = val * (1.250 / 65535.000);				// Volt = ADC Result x [1.250V / 65535]		
	printf("\rAN[1..8] : %1.3f",volt); 				// Display 3-Digit Result(0-1.25V)	    	
	 	
	//AN2
	ADC0CON2 &=  ~0x0F;	            				// Reset Select ADC Input
	ADC0CON2 |=   0x01;	            				// ADC Input = AIN2,AINCOM
	RDY0 =  0; 
	while(!RDY0);	     							// Wait ADC Complete		
	val  = ((ADC0H<<8)&0xFF00);          			// Read MSB 8-Bits 
    val |= ((ADC0M<<0)&0x00FF);   					// Read LSB 8-Bits 
	volt = val * (1.250 / 65535.000);				// Volt = ADC Result x [1.250V / 65535]		
	printf(" : %1.3f",volt); 						// Display 3-Digit Result(0-1.25V)	    	
	
	//AN3
    ADC0CON2 &=  ~0x0F;	            				// Reset Select ADC Input
	ADC0CON2 |=   0x02;	            				// ADC Input = AIN3,AINCOM
	RDY0 =  0; 
	while(!RDY0);	     							// Wait ADC Complete		
	val  = ((ADC0H<<8)&0xFF00);          			// Read MSB 8-Bits 
    val |= ((ADC0M<<0)&0x00FF);   					// Read LSB 8-Bits 
	volt = val * (1.250 / 65535.000);				// Volt = ADC Result x [1.250V / 65535]		
	printf(" : %1.3f",volt); 						// Display 3-Digit Result(0-1.25V)	    	
	
	//AN4	
	ADC0CON2 &=  ~0x0F;	            				// Reset Select ADC Input
	ADC0CON2 |=   0x03;	            				// ADC Input = AIN4,AINCOM
	RDY0 =  0;
	while(!RDY0);	     							// Wait ADC Complete		
	val  = ((ADC0H<<8)&0xFF00);          			// Read MSB 8-Bits 
    val |= ((ADC0M<<0)&0x00FF);   					// Read LSB 8-Bits 
	volt = val * (1.250 / 65535.000);				// Volt = ADC Result x [1.250V / 65535]		
	printf(" : %1.3f",volt); 						// Display 3-Digit Result(0-1.25V)	    	
	 
	//AN5	
	ADC0CON2 &=  ~0x0F;	            				// Reset Select ADC Input
	ADC0CON2 |=   0x04;	            				// ADC Input = AIN5,AINCOM
	RDY0 =  0; 
	while(!RDY0);	     							// Wait ADC Complete		
	val  = ((ADC0H<<8)&0xFF00);          			// Read MSB 8-Bits 
    val |= ((ADC0M<<0)&0x00FF);   					// Read LSB 8-Bits 
	volt = val * (1.250 / 65535.000);				// Volt = ADC Result x [1.250V / 65535]		
	printf(" : %1.3f",volt); 						// Display 3-Digit Result(0-1.25V)	    	
	
	//AN6	
    ADC0CON2 &=  ~0x0F;	            				// Reset Select ADC Input
	ADC0CON2 |=   0x05;	            				// ADC Input = AIN6,AINCOM
    RDY0 =  0; 
	while(!RDY0);	     							// Wait ADC Complete		
	val  = ((ADC0H<<8)&0xFF00);          			// Read MSB 8-Bits 
    val |= ((ADC0M<<0)&0x00FF);   					// Read LSB 8-Bits 
	volt = val * (1.250 / 65535.000);				// Volt = ADC Result x [1.250V / 65535]		
	printf(" : %1.3f",volt); 						// Display 3-Digit Result(0-1.25V)	    	
	
	//AN7
	ADC0CON2 &=  ~0x0F;	            				// Reset Select ADC Input
	ADC0CON2 |=   0x06;	            				// ADC Input = AIN7,AINCOM
	RDY0 =  0;
	while(!RDY0);	     							// Wait ADC Complete		
	val  = ((ADC0H<<8)&0xFF00);          			// Read MSB 8-Bits 
    val |= ((ADC0M<<0)&0x00FF);   					// Read LSB 8-Bits 
	volt = val * (1.250 / 65535.000);				// Volt = ADC Result x [1.250V / 65535]		
	printf(" : %1.3f",volt); 						// Display 3-Digit Result(0-1.25V)	   
	 
	//AN8
    ADC0CON2 &=  ~0x0F;	            				// Reset Select ADC Input
	ADC0CON2 |=   0x07;	            				// ADC Input = AIN8,AINCOM
	RDY0 =  0; 
	while(!RDY0);	     							// Wait ADC Complete		
	val  = ((ADC0H<<8)&0xFF00);          			// Read MSB 8-Bits 
    val |= ((ADC0M<<0)&0x00FF);   					// Read LSB 8-Bits 
	volt = val * (1.250 / 65535.000);				// Volt = ADC Result x [1.250V / 65535]		
	printf(" : %1.3f",volt); 						// Display 3-Digit Result(0-1.25V)	    		
  }	    
}

// Demo DAC Generate 1KHz Sinewave Signal
void test_dac(void)
{
  // Array[0..63] = 64 Step = Degree[0..360]/64 = 5.625/Step
  // Step Degree[0..63] = 0,5.625,11.25,16.875,...,354.375,360
  // Step Output[0..63] = (sin(x)*2048)+2048
  // Example = (sin(5.625)*2048)+2048
  //         = (0.098*2048)+2048
  //         = 2248.739 
  //         = 2048 
  //         = 0x08C8
  // Sine Function Table 
  static unsigned int code table_sine[64]=			// Sine Function Table
  {
     0x07FF, 0x08C8, 0x098E, 0x0A51, 	
	 0x0B0F, 0x0BC4, 0x0C71, 0x0D12,
 	 0x0DA7, 0x0E2E, 0x0EA5, 0x0F0D, 
	 0x0F63, 0x0FA6, 0x0FD7, 0x0FF5,
 	 0x0FFF, 0x0FF5, 0x0FD7, 0x0FA6, 
	 0x0F63, 0x0F0D, 0x0EA5, 0x0E2E,
 	 0x0DA7, 0x0D12, 0x0C71, 0x0BC4, 
	 0x0B0F, 0x0A51, 0x098E, 0x08C8,
 	 0x07FF, 0x0736, 0x0670, 0x05AD, 
	 0x04EF, 0x043A, 0x038D, 0x02EC,
 	 0x0257, 0x01D0, 0x0159, 0x00F1, 
	 0x009B, 0x0058, 0x0027, 0x0009,
 	 0x0000, 0x0009, 0x0027, 0x0058, 
	 0x009B, 0x00F1, 0x0159, 0x01D0,
 	 0x0257, 0x02EC, 0x038D, 0x043A, 
	 0x04EF, 0x05AD, 0x0670, 0x0736  
  };
  int   i = 0;

  // Initial DAC
  DACCON &= ~0x10;									// DAC Output = Pin-14(DAC)
  DACCON &= ~0x08;									// DAC = 12 Bit Mode
  DACCON |=  0x04;									// DAC Scale = 0V..+AVDD
  DACCON |=  0x02;                  				// DAC = Normal Operate
  DACCON |=  0x01;                  				// Start Enable DAC Operate

  /* Print Message to RS232 */
  printf("\nET-STAMP ADuC847.....From ETT CO.,LTD");  
  printf("\nDemo DAC Generate Sinewave on DAC Pin");  
  
  // Start of Config Timer2  For Trigger = DAC 1KHz/64Cycle    
  // Fclk = 12.582912MHz
  // 1 Machine Cycle = 79.4728nS
  // 1 KHz = 1mS / 79.4728nS = 12582.912 Cycle
  // DAC Trig = 1mS/64
  //          = 196.608 Cycle = 197 Cycle
  // Timer2 Reload = 65536-197 = 65339 Cycle
  //               = 0xFF3B
  TH2 = 0xFF;										// Start Timer2 Value
  TL2 = 0x3B;  	
  RCAP2H = 0xFF;									// Reload Timer2 Value
  RCAP2L = 0x3B;
  ET2 = 1;											// Enable Timer2 Interrupt
  EA = 1;		  									// Enable Global Interrupt
  T2CON = 0x04;										// TR2 = 1(Start Timer2)+Config Timer2 = 16Bit Auto Reload
  //End of Config Timer2

  while(1)											// Loop Continue
  {
    while(!Trig_Ready);	 							// Wait 1/64mS Trigger

    DACH = (table_sine[i] >> 8) & 0xFF; 			// Update MSB DAC Sine Output(0V..+AVDD)
	DACL = (table_sine[i] & 0xFF);					// Update LSB DAC
	i++;											// Next Pointer
	i &= 0x3F;         								// 0..63

	Trig_Ready = 0;									// Reset Trigger Status
  }

}

void test_i2c(void)
{
  const char seg_code[16]={0x3F,0x06,0x5b,0x4F,	 	// 7-Segment Code 0..9,A..F
  						   0x66,0x6D,0x7D,0x07,
						   0x7F,0x6F,0x77,0x7c,
						   0x39,0x5e,0x79,0x71};	
						  
  int  seg_ptr;										// 7-Segment Pointer Display
  unsigned char data_out;			 				// Write SPI Buffer

  // Initial I2C Interface to PCF8574A  
  MDE = 1;											// MDE = 1  => SDA(MDO),SCL(MCO) Pin = Output
  MCO = 1;											// SCL = 1
  MDO = 1;  										// SDA = 1
  I2CM =1;											// I2CM = 1 => Master Mode

  while(1)											// Loop Continue
  { 
	for(seg_ptr=0;seg_ptr<=15;seg_ptr++)			// 0..9,A..F
	{
	  data_out=seg_code[seg_ptr];					// Get Segment Code	  
	  write_PCF8574(0x70,~data_out);				// Send I2C = PCF8574A+Write(0111+000+W),LED(Toggle Logic Drive)		  
      delay (50000);	  							// Delay Display Output
    }    
  }  
}

//Demo Test GPIO Output Mode
void test_gpio_manual(void)
{
  unsigned char rs232_data;							// RS232 Get Byte
  unsigned int j;				   					// LED Output Loop Count
  P0    |= 0xFF;									// OFF P0[7..0] Output
  P2    |= 0xFF;									// OFF P2[7..0] Output
  P3    |= 0xFC;									// OFF P3[7..2] Output
 
  /* Print Message to RS232 */
  printf("\nET-STAMP ADuC847.....From ETT CO.,LTD");  
  printf("\nManual Test GPIO Port Output Sink LED"); 
  printf("\nPort P0[0..7]"); 
  printf("\nPort P2[0..7]"); 
  printf("\nPort P3[2..7]"); 
  printf("\nPress Any Key For Step Test Pin Port"); 

  while (1) 										// Loop Continue 
  {   
	/* Forward Direction */
    // Shift Left P0[7..0]                 							
    for (j=0x01; j<= 0x80; j<<=1)  					// Blink LED P0.7 <- P0.0
    {   
      P0 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0) 			 
      rs232_data = getchar(); 						// Wait Continue
    }
	P0 |= 0xFF;	 									// OFF P0[7..0] LED Output

	// Shift Left P2[7..0]                 							
    for (j=0x01; j<= 0x80; j<<=1)  					// Blink LED P2.7 <- P2.0
    {   
      P2 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0) 
      rs232_data = getchar(); 						// Wait Continue
    }
	P2 |= 0xFF;										// OFF P2[7..0] LED Output

	// Shift Left P3[7..2]                 							
    for (j=0x04; j<= 0x80; j<<=1)  					// Blink LED P3.7 <- P3.2
    {   
      P3 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0) 
      rs232_data = getchar(); 						// Wait Continue
    }
	P3 |= 0xFC;										// OFF P3[7..2] LED Output

	/* Reverse Direction */
	// Shift Right P3[7..2]
    for (j=0x80; j>= 0x04; j>>=1)  					// Blink LED P3.7 -> P3.2
    {   
      P3 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0)  
      rs232_data = getchar(); 						// Wait Continue
    }
	P3 |= 0xFC;										// OFF P3[7..2] LED Output

	// Shift Right P2[7..0]
    for (j=0x80; j>= 0x01; j>>=1)  					// Blink LED P2.7 -> P2.0
    {   
      P2 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0)  
      rs232_data = getchar(); 						// Wait Continue
    }
	P2 |= 0xFF;										// OFF P2[7..0] LED Output
	
	// Shift Right P0[7..0]
    for (j=0x80; j>= 0x01; j>>=1)  					// Blink LED P0.7 -> P0.0
    {   
      P0 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0)  
      rs232_data = getchar(); 						// Wait Continue
    }
	P0 |= 0xFF;										// OFF P0[7..0] LED Output
  }
}

//Demo Test GPIO Output Mode
void test_gpio_auto(void)
{
  unsigned int j;				   					// LED Output Loop Count
  P0    |= 0xFF;									// OFF P0[7..0] Output
  P2    |= 0xFF;									// OFF P2[7..0] Output
  P3    |= 0xFC;									// OFF P3[7..2] Output
 
  /* Print Message to RS232 */
  printf("\nET-STAMP ADuC847.....From ETT CO.,LTD");  
  printf("\nAuto Test GPIO Port Output Sink LED"); 
  printf("\nPort P0[0..7]"); 
  printf("\nPort P2[0..7]"); 
  printf("\nPort P3[2..7]"); 
  
  while (1) 										// Loop Continue 
  {   
	/* Forward Direction */
    // Shift Left P0[7..0]                 							
    for (j=0x01; j<= 0x80; j<<=1)  					// Blink LED P0.7 <- P0.0
    {   
      P0 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0) 			 
      delay (50000);	  							// Delay Display Output
    }
	P0 |= 0xFF;	 									// OFF P0[7..0] LED Output

	// Shift Left P2[7..0]                 							
    for (j=0x01; j<= 0x80; j<<=1)  					// Blink LED P2.7 <- P2.0
    {   
      P2 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0) 
      delay (50000);	  							// Delay Display Output
    }
	P2 |= 0xFF;										// OFF P2[7..0] LED Output

	// Shift Left P3[7..2]                 							
    for (j=0x04; j<= 0x80; j<<=1)  					// Blink LED P3.7 <- P3.2
    {   
      P3 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0) 
      delay (50000);	  							// Delay Display Output
    }
	P3 |= 0xFC;										// OFF P3[7..2] LED Output

	/* Reverse Direction */
	// Shift Right P3[7..2]
    for (j=0x80; j>= 0x04; j>>=1)  					// Blink LED P3.7 -> P3.2
    {   
      P3 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0)  
      delay (50000);	  							// Delay Display Output
    }
	P3 |= 0xFC;										// OFF P3[7..2] LED Output

	// Shift Right P2[7..0]
    for (j=0x80; j>= 0x01; j>>=1)  					// Blink LED P2.7 -> P2.0
    {   
      P2 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0)  
      delay (50000);	  							// Delay Display Output
    }
	P2 |= 0xFF;										// OFF P2[7..0] LED Output
	
	// Shift Right P0[7..0]
    for (j=0x80; j>= 0x01; j>>=1)  					// Blink LED P0.7 -> P0.0
    {   
      P0 = 0xFF-j;                         			// Output to LED Port(Toggle Active=0)  
      delay (50000);	  							// Delay Display Output
    }
	P0 |= 0xFF;										// OFF P0[7..0] LED Output
  }
}

/*********************************/
/* Write Output Data to PCF8574A */
/* Addr = ID Code+Direction(W:0) */
/* Data = Output Data Write Port */
/*********************************/
void write_PCF8574(unsigned char Addr,unsigned char Data)	
{
  int I2CBit = 0;									// Bit Counter

  //Start of Generate Start
  MDE = 1;	    									// SDA = Output 
  MDO = 0; 											// SDA = 0
  I2C_Delay();
  MCO = 0; 											// SCL = 0 (Start Bit)
  //End of Generate Start
    
  //Start of Send Slave Address
  for (I2CBit = 0; I2CBit<8; I2CBit++)				// 8 Bit Write
  {
    MDO = (bit)((Addr & 0x80) >> 7);				// MSB First
	MCO = 1;
	I2C_Delay();
	MCO = 0;  										// Strobe Bit Data
	Addr <<= 1;	 									// Next Bit Data
  }
  //ACK Bit
  MDE = 0;											// SDA = Inpit(Read ACK)
  MCO = 1;											// SCL = 1
  I2C_Delay();
  //Get & Verify ACK Here  
  MCO = 0;											// SCL = 0
  MDE = 1;											// SDA = Output
  //End of Send Slave Address

  //Start of Send Data
  for (I2CBit = 0; I2CBit<8; I2CBit++)				// 8 Bit Write
  {
    MDO = (bit)((Data & 0x80) >> 7);				// MSB First
	MCO = 1;

	I2C_Delay();
	MCO = 0;  										// Strobe Bit Data
	Data <<= 1;	 									// Next Bit Data
  }
  //ACK Bit
  MDE = 0;											// SDA = Inpit(Read ACK)
  MCO = 1;											// SCL = 1
  I2C_Delay();
  //Get & Verify ACK Here  
  MCO = 0;											// SCL = 0
  MDE = 1;											// SDA = Output
  //End of Send Data

  //Generate Stop Bit
  MDE = 1;	    									// SDA = Output 
  MDO = 0;  										// SDA = 0
  MCO = 1;  										// SCL = 1 
  I2C_Delay();
  MDO = 1;  										// SDA = 1 (Stop Bit)
}

/**************************/
/* Delay I2C Clock Signal */
/**************************/
void I2C_Delay(void)
{
  int x=50;  										// Short Delay Counter
  while(x > 0) {x--;}
}

/*******************************************/
/* Long Delay Time Function(1..4294967295) */
/*******************************************/
void delay(unsigned long i)
{
  while(i > 0) {i--;}								// Loop Decrease Counter	
  return;
}

//Timer2 Overflow Service Every 1/64mS
void timer2_service () interrupt 5
{
  Trig_Ready = 1;									// 1/64 Trigger Status
  TF2 = 0;											// Reset Timer2 Overflow
}
